function segmentedImage = segmentImageLFDAIter(imgPath, fgSample, bgSample)
%SEGMENTIMAGELFDAIter Segment image using iterative LFDA (I-MMFT)
%
%   Input:
%       imgPath : path to the input image
%       fgSample: pixels of foreground scribbles
%       bgSample: pixels of background scribbles
% 
%   Output:
%       segmentedImage: binary image representing the output segmentation
%
% (c) Moustafa Meshry, moustafa.meshry@alexu.edu.eg
%     Department of Compter and Systems Engineering, Alexandria University, Egypt.

    % extract features for all pixels
    imgFeatures = extractFeatures(imgPath, fgSample, bgSample);
    [height, width, featureVectorSize] = size(imgFeatures);
        
    % Create user fg/bg scribbles binary mask
    fgSampleMask = false(height, width);
    bgSampleMask = false(height, width);
    fgSampleMask(sub2ind(size(fgSampleMask), fgSample(:,1), fgSample(:,2))) = true;
    bgSampleMask(sub2ind(size(bgSampleMask), bgSample(:,1), bgSample(:,2))) = true;

    originalFgSampleMask = fgSampleMask;
    originalBgSampleMask = bgSampleMask;

    % constants for the iterative method
    kMaxIter = 5;
    kFgGrowFactor = 1.6;
    kBgGrowFactor = 1.1;
    kMaxTrainingSize = 8000;
    kDim = 6;

    ratioclass=zeros(width*height,1);
    iter = 1;
    while ((size(fgSample, 1) + size(bgSample, 1) < kMaxTrainingSize && ...
            iter <= kMaxIter) || iter == 1)

        iter = iter + 1;

        % re-extract image features based on the expanded scribbles
        clear fgSample;
        clear bgSample;
        [y, x] = find(fgSampleMask);
        fgSample = [y, x];
        [y, x] = find(bgSampleMask);
        bgSample = [y, x];
        imgFeatures(:,:,:) = extractFeatures(imgPath, fgSample, bgSample);

        % select features of user-labeled foreground
        fgSampleMask3D = repmat(fgSampleMask, 1, featureVectorSize);
        fgSampleMask3D = reshape(fgSampleMask3D, size(imgFeatures));
        fgFeatures = imgFeatures(fgSampleMask3D);
        fgFeatures = reshape(fgFeatures, [], featureVectorSize);

        % select features of user-labeled background
        bgSampleMask3D = repmat(bgSampleMask, 1, featureVectorSize);
        bgSampleMask3D = reshape(bgSampleMask3D, size(imgFeatures));
        bgFeatures = imgFeatures(bgSampleMask3D);
        bgFeatures = reshape(bgFeatures, [], featureVectorSize);

        %% LFDA Training

        trainSample = [fgFeatures; bgFeatures];
        trainLabels = ones(size(trainSample, 1), 1);
        trainLabels(size(fgFeatures, 1) + 1 : end) = 0;
        W = LFDA(trainSample', trainLabels);
        pixelsVec = reshape(imgFeatures, height * width, []);
        pixelsScores = pixelsVec * W;

        clear imgFeatures;

        % Reduce dimensionality
        pixelsScores(:, kDim + 1 : end) = [];

        posTrainScores = pixelsScores(originalFgSampleMask(:), :);
        negTrainScores = pixelsScores(originalBgSampleMask(:), :);


        %% Pixel classification
        
        [~,d1] = knnsearch(posTrainScores, pixelsScores, 'k', 2);
        [~,d2] = knnsearch(negTrainScores, pixelsScores, 'k', 2);
        clear pixelsScores;

        ratioclass=.995*ratioclass+.005*log(sum(d1,2)./sum(d2,2));
        reshapedratioclass=reshape(ratioclass, height, width);
        bgMask = reshapedratioclass > 0;
          
        %% Expanding scribbles kNN log distance ratio
        
        [sortedVals, sortedinds] = sort(ratioclass(:));
        fgCutoffIndex = floor(kFgGrowFactor * size(fgSample, 1))+1 ; 
        bgCutoffIndex = length(sortedVals) - floor(kBgGrowFactor * ...
            size(bgSample, 1)) + 1; 
        [y2, x2] = ind2sub([height, width], sortedinds(1:fgCutoffIndex));
        fgSample = [y2 x2];
        [y2, x2] = ind2sub([height, width], sortedinds(bgCutoffIndex:length(sortedVals)));
        bgSample = [y2, x2];
        fgSampleMask(sub2ind(size(fgSampleMask), fgSample(:,1), fgSample(:,2))) = true;
        bgSampleMask(sub2ind(size(bgSampleMask), bgSample(:,1), bgSample(:,2))) = true;        
    end
    
    [y, x] = find(bgMask);
    segmentedImage = repmat(255, height, width);
    segmentedImage(sub2ind(size(segmentedImage), y, x)) = 0;
    segmentedImage = postProcess(segmentedImage, originalFgSampleMask, originalBgSampleMask);
    
end